"""
sinking.py

Implementation of the Sink operator S for the tick‑flip algebra.

The sink operator moves mass one layer inward on the context lattice.
In array terms this corresponds to shifting all entries one index towards
the end of the array.  Mass initially at the innermost layer (rightmost
index) collects incoming mass from its neighbour.  No mass appears
beyond the left‑hand boundary.
"""

import numpy as np
from .tick_state import TickState


def S(state: TickState) -> TickState:
    """
    Apply one sinking (inward flip) step to a tick state.

    Parameters
    ----------
    state : TickState
        The input tick state.  Its distribution is not mutated.

    Returns
    -------
    TickState
        A new tick state with mass shifted one layer inward.  At the
        right boundary the existing mass and the mass shifting in from the
        previous layer are added together.  Mass that would fall off the
        left boundary is discarded (i.e. the leftmost entry of the
        output distribution is set to zero).
    """
    dist = state.distribution
    L = dist.size
    new = np.zeros_like(dist)
    if L == 0:
        return TickState(new, state.N)
    # Shift interior values one step towards the right: new[i] = dist[i-1]
    if L > 2:
        new[1 : L - 1] = dist[0 : L - 2]
    # At the right boundary collect mass from itself and the previous index
    if L > 1:
        new[L - 1] = dist[L - 1] + dist[L - 2]
    else:
        # Only one element; nothing to shift
        new[0] = dist[0]
    # Leftmost entry loses mass since nothing shifts into it (remains zero)
    return TickState(new, state.N)
